package com.chemastudios.ecorecolector;

import androidx.appcompat.app.AppCompatActivity;
import androidx.recyclerview.widget.RecyclerView;
import androidx.recyclerview.widget.LinearLayoutManager;

import android.os.Bundle;
import android.util.Log;
import android.widget.*;
import android.view.View;

import com.android.volley.*;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;
import com.google.android.material.button.MaterialButton;
import com.google.android.material.textfield.TextInputEditText;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DetalleSolicitudActivityV1 extends AppCompatActivity {

    private static final String TAG = "DetalleSolicitudV1";

    // Views
    private TextView txtNombreUsuario, txtDireccionUsuario, txtTelefonoUsuario;
    private TextInputEditText edtUsuarioId, edtKilos;
    private TextView txtPuntos, txtTotal;
    private Spinner spnCategoria;
    private MaterialButton btnAgregar, btnEnviar;
    private RecyclerView recyclerView;

    // Lista y adapter de productos
    private List<Producto> listaProductos = new ArrayList<>();
    private ProductoAdapterv1 adapter;

    // Variables de control
    private double puntosPorKilo = 0.5;
    private double totalPuntos = 0;
    private double totalKilos = 0;

    private String[] categorias = {"Plástico", "Cartón", "Vidrio", "Metal", "Orgánico", "Electrónicos"};
    private String idSolicitud = "";
    private String idUsuario = "";

    private static final String URL_GUARDAR = "http://74.208.75.82/guardarProductos.php";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_detalle_solicitud);

        // Vincular views
        txtNombreUsuario = findViewById(R.id.txtNombreUsuariov1);
        txtDireccionUsuario = findViewById(R.id.txtDireccionUsuario);
        txtTelefonoUsuario = findViewById(R.id.txtTelefonoUsuario);
        edtUsuarioId = findViewById(R.id.edtUsuarioId);
        edtKilos = findViewById(R.id.edtKilosv1);
        txtPuntos = findViewById(R.id.txtPuntosv1);
        txtTotal = findViewById(R.id.txtTotalv1);
        spnCategoria = findViewById(R.id.spnCategoria);
        btnAgregar = findViewById(R.id.btnAgregar);
        btnEnviar = findViewById(R.id.btnGuardarv1);
        recyclerView = findViewById(R.id.recyclerProductos);

        recyclerView.setLayoutManager(new LinearLayoutManager(this));
        adapter = new ProductoAdapterv1(listaProductos, position -> borrarProducto(position));
        recyclerView.setAdapter(adapter);

        // Spinner categorías
        ArrayAdapter<String> adapterSp = new ArrayAdapter<>(this,
                android.R.layout.simple_spinner_dropdown_item, categorias);
        spnCategoria.setAdapter(adapterSp);

        // Obtener datos del Intent
        idSolicitud = getIntent().getStringExtra("idSolicitud");
        if (idSolicitud == null || idSolicitud.trim().isEmpty()) {
            Toast.makeText(this, "Error: solicitud sin ID", Toast.LENGTH_SHORT).show();
            finish();
            return;
        }

        String nombre = getIntent().getStringExtra("nombre");
        String direccion = getIntent().getStringExtra("direccion");
        String ciudad = getIntent().getStringExtra("ciudad");
        String colonia = getIntent().getStringExtra("colonia");
        String cp = getIntent().getStringExtra("cp");
        String numeroCasa = getIntent().getStringExtra("numeroCasa");
        String telefono = getIntent().getStringExtra("telefono");
        String idUsuarioRecibido = getIntent().getStringExtra("idUsuario");

        String direccionCompleta = direccion + ", " + numeroCasa + ", " + colonia + ", " + ciudad + ", CP: " + cp;

        txtNombreUsuario.setText("Nombre: " + (nombre != null ? nombre : "Desconocido"));
        txtDireccionUsuario.setText("Dirección: " + direccionCompleta);
        txtTelefonoUsuario.setText("Teléfono: " + (telefono != null && !telefono.isEmpty() ? telefono : "No disponible"));

        // Llenar ID usuario automáticamente
        if (idUsuarioRecibido != null && !idUsuarioRecibido.isEmpty()) {
            edtUsuarioId.setText(idUsuarioRecibido);
            idUsuario = idUsuarioRecibido;
        }

        btnAgregar.setOnClickListener(v -> agregarProducto());
        btnEnviar.setOnClickListener(v -> enviarProductos());

        actualizarTextos();
    }

    // Actualiza los textos de puntos y total
    private void actualizarTextos() {
        txtPuntos.setText("Puntos: 0");
        txtTotal.setText("Total puntos: " + totalPuntos + " | Kilos: " + totalKilos);
    }

    // Agregar producto
    private void agregarProducto() {
        String kilosText = edtKilos.getText().toString().trim();
        if (kilosText.isEmpty()) {
            Toast.makeText(this, "Ingresa los kilos", Toast.LENGTH_SHORT).show();
            return;
        }

        double kilos;
        try { kilos = Double.parseDouble(kilosText); }
        catch (NumberFormatException e) { Toast.makeText(this, "Kilos inválidos", Toast.LENGTH_SHORT).show(); return; }

        if (kilos <= 0) { Toast.makeText(this, "Kilos debe ser mayor que 0", Toast.LENGTH_SHORT).show(); return; }

        double puntos = kilos * puntosPorKilo;
        String categoria = spnCategoria.getSelectedItem().toString();

        Producto p = new Producto(categoria, kilos, puntos);
        listaProductos.add(p);

        totalPuntos += puntos;
        totalKilos += kilos;

        txtPuntos.setText("Puntos: " + puntos);
        txtTotal.setText("Total puntos: " + totalPuntos + " | Kilos: " + totalKilos);

        adapter.notifyItemInserted(listaProductos.size() - 1);
        edtKilos.setText("");
        edtKilos.requestFocus();
        recyclerView.post(() -> recyclerView.smoothScrollToPosition(listaProductos.size() - 1));
    }

    // Borrar producto
    private void borrarProducto(int position) {
        if (position < 0 || position >= listaProductos.size()) return;

        Producto eliminado = listaProductos.get(position);
        totalPuntos -= eliminado.getPuntosv();
        totalKilos -= eliminado.getKilos();

        listaProductos.remove(position);
        adapter.notifyItemRemoved(position);

        txtTotal.setText("Total puntos: " + totalPuntos + " | Kilos: " + totalKilos);
    }

    // Enviar productos al servidor
    private void enviarProductos() {
        String userIdText = edtUsuarioId.getText().toString().trim();
        if (userIdText.isEmpty()) {
            Toast.makeText(this, "ID usuario no disponible", Toast.LENGTH_SHORT).show();
            return;
        }
        idUsuario = userIdText;

        if (listaProductos.isEmpty()) {
            Toast.makeText(this, "No hay productos agregados", Toast.LENGTH_SHORT).show();
            return;
        }

        try {
            JSONObject json = new JSONObject();
            json.put("id_solicitud", idSolicitud);
            json.put("id_usuario", idUsuario);
            json.put("total_puntos", totalPuntos);
            json.put("total_kilos", totalKilos);

            JSONArray array = new JSONArray();
            for (Producto p : listaProductos) {
                JSONObject obj = new JSONObject();
                obj.put("categoria", p.getCategoria());
                obj.put("kilos", p.getKilos());
                obj.put("puntos", p.getPuntosv());
                array.put(obj);
            }
            json.put("productos", array);

            enviarProductosAPI(json);

        } catch (Exception e) {
            Toast.makeText(this, "Error interno", Toast.LENGTH_SHORT).show();
            e.printStackTrace();
        }
    }
    private void enviarProductosAPI(JSONObject json) {

        JsonObjectRequest jsonObjectRequest = new JsonObjectRequest(
                Request.Method.POST,
                URL_GUARDAR,
                json,
                response -> {
                    Log.d(TAG, "RESPONSE: " + response.toString());
                    try {
                        boolean success = response.getBoolean("success");
                        String msg = response.getString("msg");
                        double totalPuntosResp = response.optDouble("total_puntos", 0);
                        double totalKilosResp = response.optDouble("total_kilos", 0);

                        Toast.makeText(this,
                                msg + "\nTotal puntos: " + totalPuntosResp + " | Total kilos: " + totalKilosResp,
                                Toast.LENGTH_LONG).show();

                        if (success) finish();

                    } catch (Exception e) {
                        e.printStackTrace();
                        Toast.makeText(this, "❌ Respuesta inválida", Toast.LENGTH_SHORT).show();
                    }
                },
                error -> {
                    Log.e(TAG, "❌ Error guardando productos: " + error.toString());
                    if (error.networkResponse != null) {
                        Log.e(TAG, "STATUS: " + error.networkResponse.statusCode);
                        Log.e(TAG, "DATA: " + new String(error.networkResponse.data));
                    }
                    Toast.makeText(this, "❌ Error guardando productos", Toast.LENGTH_LONG).show();
                }
        ) {
            @Override
            public Map<String, String> getHeaders() {
                Map<String, String> headers = new HashMap<>();
                headers.put("Content-Type", "application/json; charset=UTF-8");
                return headers;
            }
        };

        Volley.newRequestQueue(this).add(jsonObjectRequest);
    }

}
