package com.chemastudios.ecorecolector;

import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.view.LayoutInflater;
import android.view.View;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.TextView;
import android.widget.Toast;

import androidx.appcompat.app.AppCompatActivity;

import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.StringRequest;
import com.android.volley.toolbox.Volley;
import com.android.volley.DefaultRetryPolicy;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

public class MainActivity extends AppCompatActivity {

    private TextView txtPuntosAcumulados, txtKilosReciclados;

    private static final String URL_API_STATS = "http://74.208.75.82/api/get_user_stats.php";

    private final Handler handler = new Handler();
    private static final int INTERVALO_ACTUALIZACION = 5000;

    private final Runnable refrescarDatos = new Runnable() {
        @Override
        public void run() {
            cargarEstadisticasUsuario();
            handler.postDelayed(this, INTERVALO_ACTUALIZACION);
        }
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.inicio_app);

        // ✅ Fade-in moderno al abrir
        View root = findViewById(R.id.rootMain);
        Animation fade = AnimationUtils.loadAnimation(this, R.anim.fade_in_modern);
        root.startAnimation(fade);

        txtPuntosAcumulados = findViewById(R.id.txtPuntosAcumulados);
        txtKilosReciclados = findViewById(R.id.txtKilosReciclados);

        ImageView iconHistorial = findViewById(R.id.ichistoriallppp);
        iconHistorial.setOnClickListener(v -> {
            startActivity(new Intent(MainActivity.this, HistorialSolicitudesActivity.class));
        });
        ImageView iconPremios = findViewById(R.id.icpremiospppl);
        iconPremios.setOnClickListener(v ->
                startActivity(new Intent(MainActivity.this, CanjeoActivity.class))
        );

        ImageView iconPerfil = findViewById(R.id.iconProfile);
        iconPerfil.setOnClickListener(v->
                startActivity(new Intent(MainActivity.this, PerfilActivity.class))
        );

        ImageView iconMenu = findViewById(R.id.iconMenu);
        iconMenu.setOnClickListener(v ->
                startActivity(new Intent(MainActivity.this, MenuPrincipalActivity.class))
        );

        Button btnVerMasReciclaje = findViewById(R.id.btnVerMasReciclaje);
        btnVerMasReciclaje.setOnClickListener(v->
                startActivity(new Intent(MainActivity.this, VerMasReciclaje.class))
        );

        validarSesion();
        mostrarMensajeBienvenida();
        cargarEstadisticasUsuario();

        View btnSolicitar = findViewById(R.id.btnSolicitar);

            btnSolicitar.setOnClickListener(v ->
                    startActivity(new Intent(MainActivity.this, SolicitudActivity.class))
            );
    }

    @Override
    protected void onResume() {
        super.onResume();
        handler.post(refrescarDatos);
    }

    @Override
    protected void onPause() {
        super.onPause();
        handler.removeCallbacks(refrescarDatos);
    }

    private void validarSesion() {
        if (SessionManager.getUserId() <= 0) {
            redirigirALogin();
        }
    }

    private void mostrarMensajeBienvenida() {

        String nombre = SessionManager.getUserName();

        if (nombre == null || nombre.trim().isEmpty() || nombre.equalsIgnoreCase("null")) {
            nombre = "Eco Héroe";
        }

        boolean yaMostrado = getSharedPreferences("APP_PREFS", MODE_PRIVATE)
                .getBoolean("bienvenida_mostrada", false);

        if (!yaMostrado) {
            showWelcome(nombre);

            getSharedPreferences("APP_PREFS", MODE_PRIVATE)
                    .edit()
                    .putBoolean("bienvenida_mostrada", true)
                    .apply();
        }
    }

    private void redirigirALogin() {
        handler.removeCallbacks(refrescarDatos);
        Intent intent = new Intent(MainActivity.this, InicioDeSeccion.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TASK);
        startActivity(intent);
        finish();
    }

    private void cargarEstadisticasUsuario() {

        final int userId = SessionManager.getUserId();

        if (userId <= 0) {
            Toast.makeText(this, "No hay sesión activa.", Toast.LENGTH_LONG).show();
            redirigirALogin();
            return;
        }

        RequestQueue queue = Volley.newRequestQueue(this);

        StringRequest stringRequest = new StringRequest(
                Request.Method.POST,
                URL_API_STATS,
                response -> {
                    try {
                        JSONObject jsonResponse = new JSONObject(response);

                        boolean error = jsonResponse.optBoolean("error", true);

                        if (!error) {
                            JSONObject userStats = jsonResponse.optJSONObject("user");

                            if (userStats != null) {

                                String puntosStr = userStats.optString("puntos", "0");
                                txtPuntosAcumulados.setText(puntosStr);

                                String kilosStr = userStats.optString("kilos_reciclados", "0.00");

                                try {
                                    double kilosDouble = Double.parseDouble(kilosStr);
                                    txtKilosReciclados.setText(
                                            String.format(Locale.US, "%.2f KG", kilosDouble)
                                    );
                                } catch (NumberFormatException e) {
                                    txtKilosReciclados.setText("0.00 KG");
                                }
                            }
                        } else {
                            Toast.makeText(MainActivity.this, "No se pudo cargar información", Toast.LENGTH_SHORT).show();
                        }

                    } catch (Exception e) {
                        Toast.makeText(MainActivity.this, "Error al procesar datos", Toast.LENGTH_LONG).show();
                    }
                },
                error -> Toast.makeText(MainActivity.this, "Error de conexión", Toast.LENGTH_LONG).show()
        ) {
            @Override
            protected Map<String, String> getParams() {
                Map<String, String> params = new HashMap<>();
                params.put("user_id", String.valueOf(userId));
                return params;
            }
        };

        stringRequest.setRetryPolicy(new DefaultRetryPolicy(
                5000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT
        ));

        queue.add(stringRequest);
    }

    private void showWelcome(String usuario) {

        LayoutInflater inflater = LayoutInflater.from(this);
        View layout = inflater.inflate(R.layout.welcome_popup, null);

        TextView txtWelcome = layout.findViewById(R.id.txtWelcomev1);
        txtWelcome.setText("¡Bienvenido " + usuario + "!");

        FrameLayout root = findViewById(android.R.id.content);
        root.addView(layout);

        Animation animIn = AnimationUtils.loadAnimation(this, R.anim.welcome_in);
        Animation animOut = AnimationUtils.loadAnimation(this, R.anim.welcome_out);

        txtWelcome.startAnimation(animIn);

        new Handler().postDelayed(() -> {
            txtWelcome.startAnimation(animOut);

            new Handler().postDelayed(() -> {
                root.removeView(layout);
            }, 600);

        }, 1500);
    }
}
